/*
 * Copyright (c) 2009-2012 jMonkeyEngine
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 *
 * * Neither the name of 'jMonkeyEngine' nor the names of its contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package ga.view.config;

import java.awt.BorderLayout;
import java.awt.DisplayMode;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.prefs.BackingStoreException;

import javax.swing.DefaultComboBoxModel;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

import com.jme3.app.SettingsDialog;
import com.jme3.system.AppSettings;

/**
 * This is an extended {@link SettingsDialog} for the Furny app. Further doc is
 * taken from {@link SettingsDialog}.
 * 
 * <code>PropertiesDialog</code> provides an interface to make use of the
 * <code>GameSettings</code> class. The <code>GameSettings</code> object is
 * still created by the client application, and passed during construction.
 * 
 * @see AppSettings
 * @author Mark Powell
 * @author Eric Woroshow
 * @author Joshua Slack - reworked for proper use of GL commands.
 */
public final class SettingsDialog2 extends JDialog {

  public static interface SelectionListener {

    public void onSelection(int selection);
  }

  public static final int NO_SELECTION = 0;
  public static final int APPROVE_SELECTION = 1;
  public static final int CANCEL_SELECTION = 2;

  private static final Logger LOGGER = Logger.getLogger(SettingsDialog2.class
      .getName());
  private static final long serialVersionUID = 1L;

  // connection to properties file.
  private final AppSettings source;
  // Title Image
  private final URL imageFile;
  // Array of supported display modes
  private final DisplayMode[] modes;
  // Array of windowed resolutions
  private final String[] windowedResolutions = { "320 x 240", "640 x 480",
      "800 x 600", "1024 x 768", "1152 x 864", "1280 x 720", };
  // UI components
  private JCheckBox vsyncBox;
  private JCheckBox fullscreenBox;
  private JCheckBox shadowsBox;
  private JCheckBox ssaoBox;
  private JCheckBox lightScatteringBox;
  private JCheckBox hqMaterialBox;
  private JCheckBox showStatisticsBox;
  private JCheckBox showCursorBox;
  private JComboBox displayResCombo;
  private JComboBox colorDepthCombo;
  private JComboBox displayFreqCombo;
  // private JComboBox rendererCombo = null;
  private JComboBox antialiasCombo;
  private JLabel icon;
  private int selection;
  private SelectionListener selectionListener;

  private String fileToSave;

  /**
   * Constructor for the <code>PropertiesDialog</code>. Creates a properties
   * dialog initialized for the primary display.
   * 
   * @param source
   *          the <code>AppSettings</code> object to use for working with the
   *          properties file.
   * @param imageFile
   *          the image file to use as the title of the dialog;
   *          <code>null</code> will result in to image being displayed
   * @throws NullPointerException
   *           if the source is <code>null</code>
   */
  public SettingsDialog2(final AppSettings source, final String imageFile,
      final boolean loadSettings) {
    this(source, getURL(imageFile), loadSettings);
  }

  /**
   * Sets the file where the settings should be saved to.
   * 
   * @param fileToSave
   *          The file name.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setFileToSave(final String fileToSave) {
    this.fileToSave = fileToSave;
  }

  /**
   * Constructor for the <code>PropertiesDialog</code>. Creates a properties
   * dialog initialized for the primary display.
   * 
   * @param source
   *          the <code>GameSettings</code> object to use for working with the
   *          properties file.
   * @param imageFile
   *          the image file to use as the title of the dialog;
   *          <code>null</code> will result in to image being displayed
   * @param loadSettings
   * @throws JmeException
   *           if the source is <code>null</code>
   */
  public SettingsDialog2(final AppSettings source, final URL imageFile,
      final boolean loadSettings) {
    if (source == null) {
      throw new NullPointerException("Settings source cannot be null");
    }

    this.source = source;
    this.imageFile = imageFile;

    // setModalityType(Dialog.ModalityType.APPLICATION_MODAL);
    setModal(true);

    final AppSettings registrySettings = new AppSettings(true);

    String appTitle;
    if (source.getTitle() != null) {
      appTitle = source.getTitle();
    } else {
      appTitle = registrySettings.getTitle();
    }
    try {
      registrySettings.load(appTitle);
    } catch (final BackingStoreException ex) {
      LOGGER.log(Level.WARNING, "Failed to load settings", ex);
    }

    if (loadSettings) {
      source.copyFrom(registrySettings);
    } else if (!registrySettings.isEmpty()) {
      source.mergeFrom(registrySettings);
    }

    final GraphicsDevice device = GraphicsEnvironment
        .getLocalGraphicsEnvironment().getDefaultScreenDevice();

    modes = device.getDisplayModes();
    Arrays.sort(modes, new DisplayModeSorter());

    createUI();
  }

  public void setSelectionListener(final SelectionListener sl) {
    this.selectionListener = sl;
  }

  public int getUserSelection() {
    return selection;
  }

  private void setUserSelection(final int selection) {
    this.selection = selection;
    selectionListener.onSelection(selection);
  }

  /**
   * <code>setImage</code> sets the background image of the dialog.
   * 
   * @param image
   *          <code>String</code> representing the image file.
   */
  public void setImage(final String image) {
    try {
      final URL file = new URL("file:" + image);
      setImage(file);
      // We can safely ignore the exception - it just means that the user
      // gave us a bogus file
    } catch (final MalformedURLException e) {
    }
  }

  /**
   * <code>setImage</code> sets the background image of this dialog.
   * 
   * @param image
   *          <code>URL</code> pointing to the image file.
   */
  public void setImage(final URL image) {
    icon.setIcon(new ImageIcon(image));
    pack(); // Resize to accomodate the new image
    setLocationRelativeTo(null); // put in center
  }

  /**
   * <code>showDialog</code> sets this dialog as visble, and brings it to the
   * front.
   */
  public void showDialog() {
    setLocationRelativeTo(null);
    setVisible(true);
    toFront();
  }

  /**
   * <code>init</code> creates the components to use the dialog.
   */
  private void createUI() {
    try {
      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
    } catch (final Exception e) {
      LOGGER.warning("Could not set native look and feel.");
    }

    addWindowListener(new WindowAdapter() {

      @Override
      public void windowClosing(final WindowEvent e) {
        setUserSelection(CANCEL_SELECTION);
        dispose();
      }
    });

    if (source.getIcons() != null) {
      safeSetIconImages(Arrays.asList((BufferedImage[]) source.getIcons()));
    }

    setTitle("Select Display Settings");

    // The panels...
    final JPanel mainPanel = new JPanel();
    final JPanel centerPanel = new JPanel();
    final JPanel optionsPanel = new JPanel();
    final JPanel buttonPanel = new JPanel();
    // The buttons...
    final JButton ok = new JButton("Ok");
    final JButton cancel = new JButton("Cancel");

    icon = new JLabel(imageFile != null ? new ImageIcon(imageFile) : null);

    mainPanel.setLayout(new BorderLayout());

    centerPanel.setLayout(new BorderLayout());

    final KeyListener aListener = new KeyAdapter() {

      @Override
      public void keyPressed(final KeyEvent e) {
        if (e.getKeyCode() == KeyEvent.VK_ENTER) {
          if (verifyAndSaveCurrentSelection()) {
            setUserSelection(APPROVE_SELECTION);
            dispose();
          }
        } else if (e.getKeyCode() == KeyEvent.VK_ESCAPE) {
          setUserSelection(CANCEL_SELECTION);
          dispose();
        }
      }
    };

    displayResCombo = setUpResolutionChooser();
    displayResCombo.addKeyListener(aListener);
    colorDepthCombo = new JComboBox();
    colorDepthCombo.addKeyListener(aListener);
    displayFreqCombo = new JComboBox();
    displayFreqCombo.addKeyListener(aListener);
    antialiasCombo = new JComboBox();
    antialiasCombo.addKeyListener(aListener);
    fullscreenBox = new JCheckBox("Use Fullscreen");
    fullscreenBox.setSelected(source.isFullscreen());
    fullscreenBox.addActionListener(new ActionListener() {

      @Override
      public void actionPerformed(final ActionEvent e) {
        updateResolutionChoices();
      }
    });
    vsyncBox = new JCheckBox("Use VSync");
    vsyncBox.setSelected(source.isVSync());
    // rendererCombo = setUpRendererChooser();
    // rendererCombo.addKeyListener(aListener);

    shadowsBox = new JCheckBox("Use Shadows");
    shadowsBox.setSelected(source.getBoolean("UseShadows"));

    ssaoBox = new JCheckBox("Use SSAO (Experimental)");
    ssaoBox.setSelected(source.getBoolean("UseSSAO"));

    lightScatteringBox = new JCheckBox("Use Light Scattering (Experimental)");
    lightScatteringBox.setSelected(source.getBoolean("UseLightScattering"));

    hqMaterialBox = new JCheckBox("Use HQ Material (Experimental)");
    hqMaterialBox.setSelected(source.getBoolean("UseHQMaterial"));

    showStatisticsBox = new JCheckBox("Show Graphics Statistics");
    showStatisticsBox.setSelected(source.getBoolean("ShowStatistics"));

    showCursorBox = new JCheckBox(
        "Show Cursor (Can Be Disabled For Touchscreens)");
    showCursorBox.setSelected(source.getBoolean("ShowCursor"));

    updateResolutionChoices();
    updateAntialiasChoices();
    displayResCombo.setSelectedItem(source.getWidth() + " x "
        + source.getHeight());
    colorDepthCombo.setSelectedItem(source.getBitsPerPixel() + " bpp");

    optionsPanel.setLayout(new GridBagLayout());

    final GridBagConstraints constraints = new GridBagConstraints();
    constraints.gridx = 0;
    constraints.gridy = 0;
    constraints.insets = new Insets(5, 5, 0, 5);
    constraints.anchor = GridBagConstraints.WEST;

    optionsPanel.add(displayResCombo, constraints);

    constraints.gridx++;
    optionsPanel.add(colorDepthCombo, constraints);
    constraints.gridx++;
    optionsPanel.add(displayFreqCombo, constraints);
    constraints.gridx++;
    optionsPanel.add(antialiasCombo, constraints);

    constraints.gridwidth = 2;
    constraints.gridx = 0;
    constraints.gridy++;
    optionsPanel.add(fullscreenBox, constraints);
    constraints.gridx += 2;
    optionsPanel.add(vsyncBox, constraints);

    constraints.gridx = 0;
    constraints.gridy++;
    optionsPanel.add(shadowsBox, constraints);
    constraints.gridx += 2;
    optionsPanel.add(ssaoBox, constraints);

    constraints.gridx = 0;
    constraints.gridy++;
    optionsPanel.add(lightScatteringBox, constraints);
    constraints.gridx += 2;
    optionsPanel.add(hqMaterialBox, constraints);

    constraints.insets = new Insets(5, 5, 5, 5);
    constraints.gridx = 0;
    constraints.gridy++;
    optionsPanel.add(showStatisticsBox, constraints);
    constraints.gridx += 2;
    optionsPanel.add(showCursorBox, constraints);

    // optionsPanel.add(rendererCombo);

    // Set the button action listeners. Cancel disposes without saving, OK
    // saves.
    ok.addActionListener(new ActionListener() {

      @Override
      public void actionPerformed(final ActionEvent e) {
        if (verifyAndSaveCurrentSelection()) {
          setUserSelection(APPROVE_SELECTION);
          dispose();
        }
      }
    });

    cancel.addActionListener(new ActionListener() {

      @Override
      public void actionPerformed(final ActionEvent e) {
        setUserSelection(CANCEL_SELECTION);
        dispose();
      }
    });

    buttonPanel.add(ok);
    buttonPanel.add(cancel);

    if (icon != null) {
      centerPanel.add(icon, BorderLayout.NORTH);
    }
    centerPanel.add(optionsPanel, BorderLayout.SOUTH);

    mainPanel.add(centerPanel, BorderLayout.CENTER);
    mainPanel.add(buttonPanel, BorderLayout.SOUTH);

    this.getContentPane().add(mainPanel);

    pack();
  }

  /*
   * Access JDialog.setIconImages by reflection in case we're running on JRE <
   * 1.6
   */
  private void safeSetIconImages(final List<? extends Image> icons) {
    try {
      // Due to Java bug 6445278, we try to set icon on our shared owner frame
      // first.
      // Otherwise, our alt-tab icon will be the Java default under Windows.
      final Window owner = getOwner();
      if (owner != null) {
        final Method setIconImages = owner.getClass().getMethod(
            "setIconImages", List.class);
        setIconImages.invoke(owner, icons);
        return;
      }

      final Method setIconImages = getClass().getMethod("setIconImages",
          List.class);
      setIconImages.invoke(this, icons);
    } catch (final Exception e) {
      return;
    }
  }

  /**
   * <code>verifyAndSaveCurrentSelection</code> first verifies that the display
   * mode is valid for this system, and then saves the current selection as a
   * properties.cfg file.
   * 
   * @return if the selection is valid
   */
  private boolean verifyAndSaveCurrentSelection() {
    String display = (String) displayResCombo.getSelectedItem();
    final boolean fullscreen = fullscreenBox.isSelected();
    final boolean vsync = vsyncBox.isSelected();

    final int width = Integer.parseInt(display.substring(0,
        display.indexOf(" x ")));
    display = display.substring(display.indexOf(" x ") + 3);
    final int height = Integer.parseInt(display);

    final String depthString = (String) colorDepthCombo.getSelectedItem();
    int depth = -1;
    if (depthString.equals("???")) {
      depth = 0;
    } else {
      depth = Integer.parseInt(depthString.substring(0,
          depthString.indexOf(' ')));
    }

    final String freqString = (String) displayFreqCombo.getSelectedItem();
    int freq = -1;
    if (fullscreen) {
      if (freqString.equals("???")) {
        freq = 0;
      } else {
        freq = Integer
            .parseInt(freqString.substring(0, freqString.indexOf(' ')));
      }
    }

    final String aaString = (String) antialiasCombo.getSelectedItem();
    int multisample = -1;
    if (aaString.equals("Disabled")) {
      multisample = 0;
    } else {
      multisample = Integer.parseInt(aaString.substring(0,
          aaString.indexOf('x')));
    }

    // FIXME: Does not work in Linux
    /*
     * if (!fullscreen) { //query the current bit depth of the desktop int
     * curDepth = GraphicsEnvironment.getLocalGraphicsEnvironment()
     * .getDefaultScreenDevice().getDisplayMode().getBitDepth(); if (depth >
     * curDepth) { showError(this,"Cannot choose a higher bit depth in windowed
     * " + "mode than your current desktop bit depth"); return false; } }
     */

    final String renderer = "LWJGL-OpenGL2";// (String)
                                            // rendererCombo.getSelectedItem();

    boolean valid = false;

    // test valid display mode when going full screen
    if (!fullscreen) {
      valid = true;
    } else {
      final GraphicsDevice device = GraphicsEnvironment
          .getLocalGraphicsEnvironment().getDefaultScreenDevice();
      valid = device.isFullScreenSupported();
    }

    if (valid) {
      // use the GameSettings class to save it.
      source.setWidth(width);
      source.setHeight(height);
      source.setBitsPerPixel(depth);
      source.setFrequency(freq);
      source.setFullscreen(fullscreen);
      source.setVSync(vsync);
      // source.setRenderer(renderer);
      source.setSamples(multisample);

      // UseShadows(bool)=true
      // UseSSAO(bool)=true
      // UseLightScattering(bool)=false
      // UseHQMaterial(bool)=true
      // ShowStatistics(bool)=false
      // ShowCursor(bool)=true

      source.putBoolean("UseShadows", shadowsBox.isSelected());
      source.putBoolean("UseSSAO", ssaoBox.isSelected());
      source.putBoolean("UseLightScattering", lightScatteringBox.isSelected());
      source.putBoolean("UseHQMaterial", hqMaterialBox.isSelected());
      source.putBoolean("ShowStatistics", showStatisticsBox.isSelected());
      source.putBoolean("ShowCursor", showCursorBox.isSelected());

      if (fileToSave != null) {
        OutputStream os = null;
        try {
          final File path = new File(".");

          System.out.println(path.getCanonicalPath());

          final File f = new File(path, fileToSave);
          System.out.println(f.getCanonicalPath());
          f.getParentFile().mkdirs();
          os = new FileOutputStream(f);
          source.save(os);
          LOGGER.log(Level.INFO, f.getCanonicalPath() + " saved");
        } catch (final Exception ex) {
          LOGGER.log(Level.WARNING, "Failed to save setting changes", ex);
        } finally {
          try {
            os.close();
          } catch (final Exception e) {
          }
        }
      }
    } else {
      showError(
          this,
          "Your monitor claims to not support the display mode you've selected.\n"
              + "The combination of bit depth and refresh rate is not supported.");
    }

    return valid;
  }

  /**
   * <code>setUpChooser</code> retrieves all available display modes and places
   * them in a <code>JComboBox</code>. The resolution specified by GameSettings
   * is used as the default value.
   * 
   * @return the combo box of display modes.
   */
  private JComboBox setUpResolutionChooser() {
    final String[] res = getResolutions(modes);
    final JComboBox resolutionBox = new JComboBox(res);

    resolutionBox.setSelectedItem(source.getWidth() + " x "
        + source.getHeight());
    resolutionBox.addActionListener(new ActionListener() {

      @Override
      public void actionPerformed(final ActionEvent e) {
        updateDisplayChoices();
      }
    });

    return resolutionBox;
  }

  /**
   * <code>setUpRendererChooser</code> sets the list of available renderers.
   * Data is obtained from the <code>DisplaySystem</code> class. The renderer
   * specified by GameSettings is used as the default value.
   * 
   * @return the list of renderers.
   */
  private JComboBox setUpRendererChooser() {
    final String modes[] = { "NULL", "JOGL-OpenGL1", "LWJGL-OpenGL2",
        "LWJGL-OpenGL3", "LWJGL-OpenGL3.1" };
    final JComboBox nameBox = new JComboBox(modes);
    nameBox.setSelectedItem(source.getRenderer());
    return nameBox;
  }

  /**
   * <code>updateDisplayChoices</code> updates the available color depth and
   * display frequency options to match the currently selected resolution.
   */
  private void updateDisplayChoices() {
    if (!fullscreenBox.isSelected()) {
      // don't run this function when changing windowed settings
      return;
    }
    final String resolution = (String) displayResCombo.getSelectedItem();
    String colorDepth = (String) colorDepthCombo.getSelectedItem();
    if (colorDepth == null) {
      colorDepth = source.getBitsPerPixel() + " bpp";
    }
    String displayFreq = (String) displayFreqCombo.getSelectedItem();
    if (displayFreq == null) {
      displayFreq = source.getFrequency() + " Hz";
    }

    // grab available depths
    final String[] depths = getDepths(resolution, modes);
    colorDepthCombo.setModel(new DefaultComboBoxModel(depths));
    colorDepthCombo.setSelectedItem(colorDepth);
    // grab available frequencies
    final String[] freqs = getFrequencies(resolution, modes);
    displayFreqCombo.setModel(new DefaultComboBoxModel(freqs));
    // Try to reset freq
    displayFreqCombo.setSelectedItem(displayFreq);
  }

  /**
   * <code>updateResolutionChoices</code> updates the available resolutions list
   * to match the currently selected window mode (fullscreen or windowed). It
   * then sets up a list of standard options (if windowed) or calls
   * <code>updateDisplayChoices</code> (if fullscreen).
   */
  private void updateResolutionChoices() {
    if (!fullscreenBox.isSelected()) {
      displayResCombo.setModel(new DefaultComboBoxModel(windowedResolutions));
      colorDepthCombo.setModel(new DefaultComboBoxModel(new String[] {
          "24 bpp", "16 bpp" }));
      displayFreqCombo
          .setModel(new DefaultComboBoxModel(new String[] { "n/a" }));
      displayFreqCombo.setEnabled(false);
    } else {
      displayResCombo.setModel(new DefaultComboBoxModel(getResolutions(modes)));
      displayFreqCombo.setEnabled(true);
      updateDisplayChoices();
    }
  }

  private void updateAntialiasChoices() {
    // maybe in the future will add support for determining this info
    // through pbuffer
    final String[] choices = new String[] { "Disabled", "2x", "4x", "6x", "8x",
        "16x" };
    antialiasCombo.setModel(new DefaultComboBoxModel(choices));
    antialiasCombo
        .setSelectedItem(choices[Math.min(source.getSamples() / 2, 5)]);
  }

  //
  // Utility methods
  //
  /**
   * Utility method for converting a String denoting a file into a URL.
   * 
   * @return a URL pointing to the file or null
   */
  private static URL getURL(final String file) {
    URL url = null;
    try {
      url = new URL("file:" + file);
    } catch (final MalformedURLException e) {
    }
    return url;
  }

  private static void showError(final java.awt.Component parent,
      final String message) {
    JOptionPane.showMessageDialog(parent, message, "Error",
        JOptionPane.ERROR_MESSAGE);
  }

  /**
   * Returns every unique resolution from an array of <code>DisplayMode</code>s.
   */
  private static String[] getResolutions(final DisplayMode[] modes) {
    final ArrayList<String> resolutions = new ArrayList<String>(modes.length);
    for (int i = 0; i < modes.length; i++) {
      final String res = modes[i].getWidth() + " x " + modes[i].getHeight();
      if (!resolutions.contains(res)) {
        resolutions.add(res);
      }
    }

    final String[] res = new String[resolutions.size()];
    resolutions.toArray(res);
    return res;
  }

  /**
   * Returns every possible bit depth for the given resolution.
   */
  private static String[] getDepths(final String resolution,
      final DisplayMode[] modes) {
    final ArrayList<String> depths = new ArrayList<String>(4);
    for (int i = 0; i < modes.length; i++) {
      // Filter out all bit depths lower than 16 - Java incorrectly
      // reports
      // them as valid depths though the monitor does not support them
      if (modes[i].getBitDepth() < 16 && modes[i].getBitDepth() > 0) {
        continue;
      }

      final String res = modes[i].getWidth() + " x " + modes[i].getHeight();
      final String depth = modes[i].getBitDepth() + " bpp";
      if (res.equals(resolution) && !depths.contains(depth)) {
        depths.add(depth);
      }
    }

    if (depths.size() == 1 && depths.contains("-1 bpp")) {
      // add some default depths, possible system is multi-depth supporting
      depths.clear();
      depths.add("24 bpp");
    }

    final String[] res = new String[depths.size()];
    depths.toArray(res);
    return res;
  }

  /**
   * Returns every possible refresh rate for the given resolution.
   */
  private static String[] getFrequencies(final String resolution,
      final DisplayMode[] modes) {
    final ArrayList<String> freqs = new ArrayList<String>(4);
    for (int i = 0; i < modes.length; i++) {
      final String res = modes[i].getWidth() + " x " + modes[i].getHeight();
      String freq;
      if (modes[i].getRefreshRate() == DisplayMode.REFRESH_RATE_UNKNOWN) {
        freq = "???";
      } else {
        freq = modes[i].getRefreshRate() + " Hz";
      }

      if (res.equals(resolution) && !freqs.contains(freq)) {
        freqs.add(freq);
      }
    }

    final String[] res = new String[freqs.size()];
    freqs.toArray(res);
    return res;
  }

  /**
   * Utility class for sorting <code>DisplayMode</code>s. Sorts by resolution,
   * then bit depth, and then finally refresh rate.
   */
  private class DisplayModeSorter implements Comparator<DisplayMode> {

    /**
     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
     */
    @Override
    public int compare(final DisplayMode a, final DisplayMode b) {
      // Width
      if (a.getWidth() != b.getWidth()) {
        return (a.getWidth() > b.getWidth()) ? 1 : -1;
      }
      // Height
      if (a.getHeight() != b.getHeight()) {
        return (a.getHeight() > b.getHeight()) ? 1 : -1;
      }
      // Bit depth
      if (a.getBitDepth() != b.getBitDepth()) {
        return (a.getBitDepth() > b.getBitDepth()) ? 1 : -1;
      }
      // Refresh rate
      if (a.getRefreshRate() != b.getRefreshRate()) {
        return (a.getRefreshRate() > b.getRefreshRate()) ? 1 : -1;
      }
      // All fields are equal
      return 0;
    }
  }

  public static void main(final String[] args) {

    SwingUtilities.invokeLater(new Runnable() {
      @Override
      public void run() {
        final AppSettings settings = new AppSettings(true);

        final SettingsDialog2 sd = new SettingsDialog2(settings, (URL) null,
            false);
        sd.setFileToSave("config/test.properties");
        final SelectionListener sl = new SelectionListener() {

          @Override
          public void onSelection(final int selection) {

            sd.dispose();

            if (selection == APPROVE_SELECTION) {
              final SettingsDialog2 sd = new SettingsDialog2(settings,
                  (URL) null, false);

              sd.setSelectionListener(this);
              sd.showDialog();
            }
          }
        };
        sd.setSelectionListener(sl);
        sd.showDialog();
      }
    });
  }
}
